/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iDate vec4(0.0,0.0,0.0,iGlobalTime)
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// Gravitation lensing simulation using DF-tracing
// by Tomkh '20.01.2015
// http:/moonedit.com/tom

#define ptcnt 12
vec3 pt[ptcnt];
float ptrad = 0.1;

void setup_scene()
{
    float t0 = iDate.w*.5;
    for(int i=0; i<ptcnt; ++i) 
    {
        float t = t0 + float(i)*1.7;
        
        // Simple animation of spheres:
        pt[i].x = cos(t)*.5;
        pt[i].y = sin(t*1.1)*.5;
        
        // Put 6 sphere in front (lens) and 6 in the back:
        pt[i].z = ((i*2<ptcnt)?4.0:16.0) + cos(t*2.1);
    }
}

vec4 df(vec3 p)
{
    float dmin = 1e32;
    vec3 n = vec3(0,0,0);
    for(int i=0; i<ptcnt; ++i) 
    {
        vec3 dp = pt[i] - p;
        float d = dot(dp,dp);
        
        // Here is the key to lensing effect, approx. gravity field:
        n += dp/d;
        
        dmin = min(d, dmin);
    }
    return vec4(n,sqrt(dmin));
}

vec4 tex(vec3 p)
{
    float dmin = 1e32;
    vec3 dpmin;
    for(int i=0; i<ptcnt; ++i) 
    {
        vec3 dp = pt[i] - p;
        float d = dot(dp,dp);
        if (d < dmin) 
        {
            dmin = d;
            dpmin = dp;
        }
    }
    float d = sqrt(dmin);
    return vec4(dpmin/d,d);
}

vec4 trace(vec3 p, vec3 n)
{
    float falloff = 1.0 - (1.0 - n.z)*16.0;
    
    // Lensing animation:
    float sc = sin(iDate.w*.1);
    sc *= sc;
    sc *= .008;
    
    // DF tracing with lens-effect here:
    vec4 dn;
    float lit = 0.0;
    for(int k=0; k<64; ++k)
    {
        dn = df(p);
        float d = dn.w;
        float surf_dist = d - ptrad;
        lit += 0.003/(d + 0.1);
        
        // Move half the distance only:
        float f = surf_dist*.5;
        p += n*f;
        
        // Modify ray direction by gravity field:
        n = normalize(n + dn.xyz*f*sc);
    }
    
    // Shading:
    vec4 norm = tex(p);
    float shade = max(0.0, 1.0 - (norm.w - ptrad)*256.0);
    vec4 col = texture2D(iChannel1, norm.xy*0.1);
    vec4 bkg = texture2D(iChannel1, n.xy*2.0)*falloff;
    return vec4(vec3(lit,lit,lit*0.6) - vec3(0.0,col.x*col.x*0.5,col.x)*shade,1.0)
         + vec4(bkg.xy,bkg.z*2.0,0.0)*0.2*(1.0-shade);
}

void main(void)
{
    setup_scene();
	vec2 uv = (gl_FragCoord.xy - iResolution.xy*0.5) / iResolution.x;
    gl_FragColor = trace(vec3(0,0,0), normalize(vec3(uv,1)));
}